import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.lang.Math;

class BinaryMatrix
{
	private boolean[][] arr;

	private int[] gaps;

	public int get(int i, int j)
	{
		return this.arr[i][j] ? 1 : 0;
	}

	/**
	 * Reassigns new values to columns if they are already satisfied (gap<=0).
	 */
	public void reassignSatisfiedColumns(float a, float b)
	{
		boolean foundSatisfiedColumns = true;

		while (foundSatisfiedColumns)
		{
			foundSatisfiedColumns = false;
			for (int c = 0; c < this.getNoColumns(); c++)
			{
				if ( this.gaps[c] <= 0 )
				{
					foundSatisfiedColumns = true;
					float den = a + (float) (Math.random() * (b - a));
					for (int r = 0; r < this.getNoRows(); r++)
					{
						this.arr[r][c] = Math.random() <= den ? true : false;
					}
				}
			}
			calcGaps();
		}
	}

	/**
	 * Remove columns if they are already satisfied (gap<=0).
	 */
	public void removeSatisfiedColumns()
	{
		boolean foundSatisfiedColumns = true;

		while (foundSatisfiedColumns)
		{
			foundSatisfiedColumns = false;
			for (int c = 0; c < this.getNoColumns(); c++)
			{
				if ( this.gaps[c] <= 0 )
				{
					foundSatisfiedColumns = true;
					
					break;
				}
			}
			calcGaps();
		}
	}
	
	public boolean isSolutionOfSize(int k)
	{
		boolean[] selectedRows = new boolean[this.getNoRows()];
		Arrays.fill(selectedRows, 0, k, true);
		Arrays.fill(selectedRows, k, this.getNoRows(), false);

		boolean s = true;
		while (s)
		{
			if ( isSolution(selectedRows) )
				return true;

			s = false;
			for (int i = this.getNoRows() - 2; i >= 0; i--)
			{
				if ( selectedRows[i] == true && selectedRows[i + 1] == false )
				{
					s = true;
					selectedRows[i] = false;
					selectedRows[i + 1] = true;

					int freeIndex = i + 2;
					for (int j = i + 2; j < this.getNoRows(); j++)
					{
						if ( selectedRows[j] == true )
						{
							selectedRows[j] = false;
							selectedRows[freeIndex++] = true;
						}
					}
					break;
				}
			}
		}
		return false;
	}

	private boolean isSolution(boolean[] selectedRows)
	{
		for (int c = 0; c < this.getNoColumns(); c++)
		{
			int t = 0;
			for (int r = 0; r < this.getNoRows(); r++)
			{
				if ( selectedRows[r] == true && this.arr[r][c] == false )
				{
					t++;
				}
			}
			if ( this.getGapForColumn(c) - t > 0 )
				return false;
		}
		return true;
	}

	int getNoRows()
	{
		return this.arr.length;
	}

	public BinaryMatrix(int noRows, int noColumns, int minGap, int maxGap)
	{
		final int maj = noRows / 2 + 1;
		if ( minGap <= 0 || minGap > maxGap || maxGap>maj)
			throw new IllegalArgumentException("");
		
		
		this.arr = new boolean[noRows][noColumns];
		for (int c = 0; c < noColumns; c++)
		{
			ArrayList<Boolean> boolArr = new ArrayList<Boolean>();
			int gap = minGap + (int) Math.round((Math.random() * (maxGap - minGap)));
			for (int r = 0; r < noRows; r++)
				boolArr.add(r < (maj - gap) ? true : false);
			Collections.shuffle(boolArr);
			for (int i = 0; i < noRows; i++)
				this.arr[i][c] = boolArr.get(i);
			calcGaps();
			if (this.gaps[c]!=gap)
				throw new IllegalAccessError();
		}
		calcGaps();
	}

	public BinaryMatrix(int noRows, int noColumns, float a, float b)
	{
		if ( a < 0f || b < 0f || b < a || a > 1f || b > 1f )
			throw new IllegalArgumentException("");

		this.arr = new boolean[noRows][noColumns];
		for (int i = 0; i < noRows; i++)
		{
			float den = a + (float) (Math.random() * (b - a));

			for (int j = 0; j < this.arr[i].length; j++)
				this.arr[i][j] = Math.random() <= den ? true : false;
		}
		calcGaps();
	}

	public BinaryMatrix(String filename, Integer expectedK) throws IOException
	{
		int k=-1;
		int m=-1;
		int n=-1;
		
		BufferedReader reader = new BufferedReader(new FileReader(filename));
		String line = null;
		
		
		// Read instance header
		if ((line = reader.readLine()) != null) {
			if(!(line.startsWith("#"))){
				System.err.println("ERROR: No instance header found.");
				System.exit(-2);
			}
			String[] headerparts = line.split("\\s*,\\s*");
			for (int h=0; h < headerparts.length; h++)
			{
				if(headerparts[h].contains("k")){
					String kValue = headerparts[h].substring(headerparts[h].indexOf('=')+1);
					k = Integer.valueOf(kValue);
				}
				if(headerparts[h].contains("m")){
					String mValue = headerparts[h].substring(headerparts[h].indexOf('=')+1);
					m = Integer.valueOf(mValue);
				}
				if(headerparts[h].contains("n")){
					String nValue = headerparts[h].substring(headerparts[h].indexOf('=')+1);
					n = Integer.valueOf(nValue);
				}
			}
		}
		if(Math.min(m,n)<1){
			System.err.println("ERROR: Invalid matrix dimentions in header.");
			System.exit(-2);
		}
		
		// Create matrix
		this.arr = new boolean[n][m];
		
	    // Read matrix
		int rowscount=0;
		while ((line = reader.readLine()) != null) {
			if(!(line.startsWith("#")))
			{
		    String[] cells = line.split("\\s");
		    if(cells.length != m)
		    {
		    	System.err.print("ERROR: Invalid number of cells in row");
		    	System.err.print(rowscount+1);
		    	System.err.println(".");
		    	System.exit(-2);
		    } else
		    {
		    	for(int i=0; i < cells.length; i++)
		    	{
		    		this.arr[rowscount][i] = Boolean.valueOf(Integer.valueOf(cells[i])==1);
		    	}
		    }
			rowscount++;
			}
		}
		
		if(rowscount != n)
		{
	    	System.err.println("ERROR: Invalid number of rows.");
			System.exit(-1);
		}
		
		reader.close();
		calcGaps();
		expectedK=k;
	}
	
	
	private void calcGaps()
	{
		final int noColumn = this.arr[0].length;
		final int rows = this.arr.length;
		this.gaps = new int[noColumn];
		Arrays.fill(this.gaps, 0);
		for (int c = 0; c < noColumn; c++)
		{

			for (int j = 0; j < rows; j++)
			{
				this.gaps[c] += this.arr[j][c] ? 1 : 0;
			}
		}
		int maj = rows / 2 + 1;
		for (int i = 0; i < this.gaps.length; i++)
			this.gaps[i] = maj - this.gaps[i];
	}

	@Override
	public String toString()
	{
		String str = "";
		for (int i = 0; i < this.arr.length; i++)
		{
			str += (i != 0) ? "\n" : "";
			for (int j = 0; j < this.arr[i].length; j++)
			{
				str += ((j != 0) ? " " : "") + (this.arr[i][j] ? "1" : "0");
			}
		}
		str += "\n";
		return str;
	}

	public int getGapForColumn(int i)
	{
		return this.gaps[i];
	}

	public void reduceSatisfiedColumns()
	{
		final int columns = this.arr[0].length;
		final int rows = this.arr.length;

		int satisfiedColums = 0;
		for (int i = 0; i < columns; i++)
		{
			if ( this.getGapForColumn(i) <= 0 )
				satisfiedColums++;
		}

		if ( satisfiedColums > 0 )
		{
			boolean[][] tmp = new boolean[rows][columns - satisfiedColums];
			for (int i = 0; i < rows; i++)
			{
				int tmpColumn = 0;
				for (int c = 0; c < columns; c++)
				{
					if ( this.getGapForColumn(c) > 0 )
					{
						tmp[i][tmpColumn++] = this.arr[i][c];
					}
				}
			}
			this.arr = tmp;
		}
		calcGaps();
	}

	public int getNoColumns()
	{
		return this.arr[0].length;
	}

	public int[] getGaps()
	{
		return this.gaps;
	}
}