#include <assert.h>
#include <stdio.h>

#include "matrix.hh"

cMatrix::cMatrix(const cMatrix& orig)
  : m_nCols(orig.m_nCols), m_nRows(orig.m_nRows),
    m_data(new vector<int>*[orig.m_nCols]),
    m_gaps(new int[orig.m_nCols]) {
  // copy data
  for (unsigned col = 0; col < m_nCols; col++) {
    m_data[col] = new vector<int>(m_nRows);
    *m_data[col] = *(orig.m_data[col]);
  }

  // copy gaps
  for (unsigned col = 0; col < m_nCols; col++)
    m_gaps[col] = orig.m_gaps[col];
}

cMatrix::cMatrix(unsigned nRows, unsigned nCols)
  : m_nCols(nCols), m_nRows(nRows), m_data(new vector<int>*[nCols]),
    m_gaps(new int[nCols]) {
  assert(nRows > 0);
  assert(nCols > 0);

  // set up data structures
  for (unsigned col = 0; col < nCols; col++)
    m_data[col] = new vector<int>(nRows);
}

cMatrix::~cMatrix() {
  // delete remaining columns; rember that some may have been deleted earlier
  for (unsigned col = 0; col < m_nCols; col++)
    delete m_data[col];

  delete [] m_gaps;
  delete [] m_data;
}

void cMatrix::resetMatrix(void){
  for (unsigned col = 0; col < m_nCols; col++)
    for (unsigned row = 0; row < m_nRows; row++)
      (*m_data[col])[row] = 0;
}

void cMatrix::setElem(unsigned row, unsigned col, int val) {
  assert(row < m_nRows);
  assert(col < m_nCols);
  (*m_data[col])[row] = (val == 0 ? 0 : 1);
}

void cMatrix::setCol(unsigned col, const vector<int>& data){
  assert(col < m_nCols);
  (*m_data[col]) = data;
}

void cMatrix::computeGaps() {
  // the threshold is defined as ceil((nRows + 1) / 2) == nRows / 2 + 1
  // the gap of a column is defined as (threshold - #of 1s in col)
  const int threshold = m_nRows / 2 + 1;
  for (unsigned col = 0; col < m_nCols; col++) {
    m_gaps[col] = threshold;
    for (unsigned row = 0; row < m_nRows; row++)
      m_gaps[col] -= getElem(row, col);
  }
}

void cMatrix::purgeSatisfiedCols() {
  // gather columns with positive gaps, delete others 
  vector<int>** newData = new vector<int>*[m_nCols];
  unsigned newCols = 0;
  for (unsigned col = 0; col < m_nCols; col++) {
    if (m_gaps[col] > 0) {
      m_gaps[newCols] = m_gaps[col];
      newData[newCols++] = m_data[col];
    } else
      delete m_data[col];
  }
  delete [] m_data;
  m_nCols = newCols;
  m_data = newData;

}

void cMatrix::bribeRow(unsigned row) {
  // if a 0 is flipped to 1, then modify the gap accordingly
  for (unsigned col = 0; col < m_nCols; col++) {
    if (getElem(row, col) == 0) {
      setElem(row, col, 1);
      m_gaps[col]--;
    }
  }
}

void cMatrix::dump() const {
  fprintf(stderr, "nRows: %u, nCols: %u\n", m_nRows, m_nCols);

  for (unsigned row = 0; row < m_nRows; row++) {
    for (unsigned col = 0; col < m_nCols; col++)
      fprintf(stderr, "%u ", getElem(row, col));
    fprintf(stderr, "\n");
  }

  fprintf(stderr, "gaps:\n");
  for (unsigned col = 0; col < m_nCols; col++)
    fprintf(stderr, "%d ", m_gaps[col]);
  fprintf(stderr, "\n");
}

void cMatrix::gaps() const {
  int max_gap = 0;
  for (unsigned col = 0; col < m_nCols; col++){
    if(max_gap < m_gaps[col]) max_gap = m_gaps[col];
  }
  
  fprintf(stdout, "%d ", max_gap);

  for (unsigned col = 0; col < m_nCols; col++){
    fprintf(stdout, "%d ", m_gaps[col]);
  }
  fprintf(stdout,"\n");
}

bool cMatrix::has_identical_col(){
  bool identical = true;
  // fprintf(stdout, "nRows, nCols: %d, %d\n", m_nRows, m_nCols);
  for (unsigned col = 0; col < m_nCols-1; col++){
    for (unsigned col2 = col+1; col2 < m_nCols; col2++){
      identical = true;
      for (unsigned row = 0; row < m_nRows; row++){
	//	fprintf(stdout, "row,col1,col2,value1,value2: %d, %d, %d, %d, %d\n",row,col,col2,getElem(row, col),getElem(row, col2));
	if(getElem(row, col) != getElem(row, col2)){
	  identical = false;
	  break;
	}  
      }
      if(identical == true){
	fprintf(stdout, "col %d and col %d are identical!\n", col, col2);
	return identical;
      }
      // End for
    } // End for
  }// End for
  //  fprintf(stdout, "identical?%d\n",identical);
  return identical;
}
