// main code
// author: Jiehua Chen

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <string.h>

#include <list>
#include <boost/smart_ptr.hpp>
using namespace boost;

#include "matrix.hh"

vector<list<unsigned> > ctypes;
vector<vector<int> > col_types;

void print_types(const vector<list<unsigned> >& types){
  vector<list<unsigned> >::const_iterator iter;
  list<unsigned>::const_iterator iter1;
  fprintf(stdout, "PRINTING CURRENT COLUMN TYPES OF SIZE %lu\n", types.size());
  for(iter = types.begin(); iter != types.end(); iter++){
    for(iter1 = (*iter).begin(); iter1 != (*iter).end(); iter1++){
      fprintf(stdout, "%u ", *iter1);
    }
    fprintf(stdout, "\n-------\n");
  }
  fprintf(stdout, "END\n");
}

void print_type(const list<unsigned>& type){
  list<unsigned>::const_iterator iter1;
  for(iter1 = type.begin(); iter1 != type.end(); iter1++){
      fprintf(stdout, "%u ", *iter1);
  }
  fprintf(stdout, "\n");
}

// return all types of size 'size' binary columns with exactly 'nOnes' 1s
void all_col_types(int nOnes, int size, 
		   list<unsigned>& ctype){
  if(nOnes <= 0){ 
    ctypes.push_back(ctype);
    return;
  }
  
  if(nOnes >= size){
    for(unsigned i = 0; i < (unsigned) size; i++){
      ctype.push_back(i);
    }
    ctypes.push_back(ctype);
    return;
  }
  size--;
  list<unsigned> copy(ctype);
  // Recursion with cases:
  // case 0: (ones, n) = (ones, n-1) 
  all_col_types(nOnes, size, copy);
  // case 1: (ones, n) = (ones-1, n-1) + [n-1]
  ctype.push_back(size);
  all_col_types(nOnes-1, size, ctype);
}

void create_col_types(unsigned nRows){
  list<unsigned>::const_iterator it;
  for(unsigned i = 0; i < ctypes.size(); i++){
    vector<int> ctype (nRows, 0);
    for(it = ctypes[i].begin(); it != ctypes[i].end(); it++){
      if(*it >= nRows) continue;
      ctype[*it] = 1;
    }
    col_types.push_back(ctype);
  }
}

// simple algorithm that will always choose the first row with the highest
// number of zeroes
// returns list of bribed voters
list<unsigned> stupid(cMatrix matrix) {
  // official plan
  // 0) clear bribe list
  // 1) while true
  // 2)     compute gaps for all columns
  // 3)     remove satisfied columns (gap <= 0)
  // 4)     if there are no more columns, finish
  // 5)     find row with the highest number of zeroes
  // 6)     set this row to 1, append to bribe list
  list<unsigned> bribes;

  // compute gaps once
  matrix.computeGaps();
  
  while (true) {
    // compute gaps
    //matrix.computeGaps();
    // remove all columns with gaps <= 0
    matrix.purgeSatisfiedCols();

    // dump remaining matrix (for debugging):
    //matrix.dump();
    
    // done?
    if (matrix.getNCols() == 0) break;

    // find row with highest number of zeroes
    unsigned bestRow = matrix.getNRows();
    unsigned bestZeros = 0;
    for (unsigned row = 0; row < matrix.getNRows(); row++) {
      // count zeros
      unsigned zeros = 0;
      for (unsigned col = 0; col < matrix.getNCols(); col++) {
        if (matrix.getElem(row, col) == 0)
          zeros++;
      }
      // better than current best?
      if (zeros > bestZeros) {
        bestZeros = zeros;
        bestRow = row;
      }
    }
    assert(bestRow != matrix.getNRows());

    // bribe it; gaps will still be up to date
    //fprintf(stderr, "bribing voter %u\n", bestRow);
    matrix.bribeRow(bestRow);
    bribes.push_back(bestRow);    
  }
  
  return bribes;
}

// return list of bribed voters
list<unsigned> greedy(cMatrix matrix) {
  // official plan (slightly modified):
  // 0) clear bribe list
  // 1) while true
  // 2)     compute gaps for all columns
  // 3)     remove satisfied columns (gap <= 0)
  // 4)     if there are no more columns, finish
  // 5)     sort columns by descending gap values
  // 6)     for each occuring gap value (a.k.a gap class)
  // 7)         for each row
  // 8)             count columns which have the current gap and a 0 in this row
  // 9)     find row which has most zeroes in highest gap class, using zeroes
  //        in lower gap classes for breaking ties; if two rows are the same,
  //        then the first is picked
  // 10)    set this row to 1, append it to bribe list
  list<unsigned> bribes;

  // compute gaps once
  matrix.computeGaps();
  
  while (true) {
    // compute gaps
    //matrix.computeGaps();
    // remove all columns with gaps <= 0
    matrix.purgeSatisfiedCols();

    // dump remaining matrix (for debugging):
    //   matrix.dump();
    
    // done?
    if (matrix.getNCols() == 0) break;

    // we actually omitt step 5 because it is not necessary
    
    // compute gap classes, sort in descending order
    list<int> gapClasses;
    for (unsigned col = 0; col < matrix.getNCols(); col++)
      gapClasses.push_back(matrix.getGap(col));
    gapClasses.sort();
    gapClasses.unique();
    gapClasses.reverse();
    
    // start with full list of rows, remove them until only one remains (or we
    // run out of tiebreakers)
    list<pair<unsigned, unsigned> > rows;
    for (unsigned row = 0; row < matrix.getNRows(); row++)
      rows.push_back(pair<unsigned, unsigned>(row, 0));

    // iterate through gap classes in descending order; for each gap classes,
    // do: 1) for each row remaining in rows, count the zeroes this row has in
    // columns of the current gap class
    //     2) find the highest number of zeroes
    //     3) eliminate all entries for rows with a lesser number of zeroes
    list<int>::const_iterator gapIter;
    for (gapIter = gapClasses.begin(); gapIter != gapClasses.end(); gapIter++) {
      // 1) for each row, count the relevant zeroes; also, keep track of the
      // highest number of zeroes
      list<pair<unsigned, unsigned> >::iterator rowIter;
      unsigned maxZeroes = 0;
      for (rowIter = rows.begin(); rowIter != rows.end(); rowIter++) {
        const unsigned row = (*rowIter).first;
        unsigned zeroes = 0;
        for (unsigned col = 0; col < matrix.getNCols(); col++) {
          if (matrix.getGap(col) == *gapIter && matrix.getElem(row, col) == 0)
            zeroes++;
        }
        (*rowIter).second = zeroes;
        // 2) keep track of maximum
        if (zeroes > maxZeroes) maxZeroes = zeroes;
      }
      // 3) remove all row entries which have not the highest number of zeroes
      for (rowIter = rows.begin(); rowIter != rows.end(); ) {
        assert((*rowIter).second <= maxZeroes);
        if ((*rowIter).second < maxZeroes)
          rowIter = rows.erase(rowIter);
        else rowIter++;
      }
      // if only one row remains, break
      if (rows.size() == 1) break;
    }
    assert(!rows.empty());

    // flip first remaining rows to all 1s, append to bribe list; gaps will be
    // updated automagically
    // fprintf(stderr, "bribing voter %u\n", rows.front().first);
    matrix.bribeRow(rows.front().first);
    bribes.push_back(rows.front().first);
  }
  
  return bribes;
}


bool brute_force(auto_ptr<vector<int> >& list, 
		 const cMatrix& bmatrix,
		 const unsigned row_size, const unsigned col_size){
  if(row_size <=0 || col_size <=0 ){
    assert(0);
  }

  // bmatrix.dump();
  for(unsigned col = 0; col < col_size; col++){
    int gap = bmatrix.getGap(col);
    for(unsigned i = 0; i < (*list).size(); i++){
      if(bmatrix.getElem((*list)[i], col) == 0) gap--;
    }
    if(gap > 0) return false;
  }
  
  return true;
}

int brute_force_subsets(int start, int counter, 
                        auto_ptr<vector<int> >& list, int range,
                        const cMatrix& bmatrix,
                        const unsigned row_size, const unsigned col_size){
 
  int new_start = 0;
  if(start > 0){
    new_start = (*list)[start-1]+1;
  }
  if(start >= counter){
    fprintf(stderr, "start: %d, counter: %d\n", start, counter);
    assert(0);
  }

  int size = -1;
  for((*list)[start] = new_start; 
      (*list)[start] <range; 
      ((*list)[start])++){
    if( start == counter-1 ){
      // fprintf(stderr, "Trying ");
      // for(unsigned i = 0; i < (*list).size(); i++){
      //   fprintf(stderr, "%d ", (*list)[i]); 
      // }
      // fprintf(stderr, "\n");
      if( brute_force(list, bmatrix, row_size, col_size) ){
	// fprintf(stderr, "Brute_force: success with %lu votes\n",
        //         (*list).size());
        //        fprintf(stderr, ". OK! \n");
        return (int) (*list).size();
      }
      //   return -1;
    }else{
      size = brute_force_subsets(start+1, counter, list, range, 
                                 bmatrix, row_size, col_size);
      if(size >= 0) return size; 
    }
  }
  return -1;
}

void create_matrix(cMatrix& matrix, const list<unsigned>& mtype,
                   const int& s, const unsigned& nCols){
  list<unsigned>::const_iterator it;
  unsigned c = 0;
  for(it = mtype.begin(); it != mtype.end(); it++){
    // fprintf(stderr, "1Checking c:%d\n", c);
    matrix.setCol(c, col_types[*it]);
    c++;
  }
  if( s > 0){      
    list<unsigned>::const_iterator type_it;
    for(; c < nCols; c++){
      if(c >=  col_types.size()) break;
      //  fprintf(stderr, "2Checking c:%d\n", c);
      matrix.setCol(c, col_types[c]);
    }
  }
}

void check_one_matrix(cMatrix& matrix) {
  matrix.computeGaps();
  // fprintf(stderr, "Checking matrix: \n");
  // matrix.dump();
  //cMatrix copy(matrix);
  int bf = -1;
  for(unsigned i = 1; i <= matrix.getNRows() / 2 + 1; i++){
    auto_ptr<vector<int> > tmp (new vector<int>);
    (*tmp).clear();
    (*tmp).resize(i);
    bf = brute_force_subsets(0, i, tmp, matrix.getNRows(), matrix,
                             matrix.getNRows(), matrix.getNCols());
    if(bf >= 0)
      break;
  }
  
  //matrix.dump();
  const list<unsigned> bribes = greedy(matrix);
  // fprintf(stderr, "greedy algorithm needs %lu bribes\n", bribes.size());
  
  if (bf != (int) bribes.size()){
    fprintf(stderr, "Greedy alg. not optimal for matrix\n");
    //cMatrix copy(matrix.getNRows(), matrix.getNCols());
    //create_matrix(copy, mtype, s, matrix.getNCols());
    cMatrix copy(matrix);
    copy.computeGaps();
    copy.dump();
    fprintf(stderr, "GD needs %lu bribes while optimal needs %d ones\n",
            bribes.size(), bf);
  }else{
    fprintf(stderr, "Greedy alg. is optimal and needs %d rows\n", bf);
  }
}

void all_matrix_types(const unsigned& nRows, const unsigned& nCols, 
		      int s, int pivot, list<unsigned>& mtype){
  // fprintf(stderr, "nRows: %d, nCols: %d, s: %d, pivot: %d\n",
  //         nRows, nCols, s, pivot);
  if( s <= 0 || s >= pivot ){
    cMatrix matrix(nRows, nCols);
    create_matrix(matrix, mtype, s, nCols);

    check_one_matrix(matrix);
    
    return;
  }

  pivot--;
  list<unsigned> copy(mtype);

  all_matrix_types(nRows, nCols, s, pivot, copy);
  mtype.push_back(pivot);
  all_matrix_types(nRows, nCols, s-1, pivot, mtype);
}

cMatrix read_matrix_from_file(FILE* f) {
  // read dimensions
  unsigned n, m;
  fscanf(f, "m=%u,n=%u", &m, &n);
  //fprintf(stderr, "reading %u rows x %u cols matrix\n", n, m);

  // read data
  cMatrix matrix(n, m);
  for (unsigned y = 0; y < n; y++) {
    for (unsigned x = 0; x < m; x++) {
      int v;
      fscanf(f, "%d", &v);
      matrix.setElem(y, x, v);
    }
  }

  return matrix;
}


int main(int argc, char* argv[]) {
  // get matrix file name, open file
  if (argc < 2 || argc > 4) {
    fprintf(stderr, "usage: ols [-s] [-g] filename\n");
    fprintf(stderr, "       -s: select stupid greedy algorithm\n");
    fprintf(stderr, "       -g: also output the rows to be bribed\n");
    return 1;
  }

  bool useStupid = false;
  if (!strcmp(argv[1], "-s")) {
    useStupid = true;
    argv++;
    argc--;
  }
  
  bool gossip = false;
  if (!strcmp(argv[1], "-g")) {
    gossip = true;
    argv++;
    argc--;
  }

  // sanity check!
  if (argc != 2) {
    fprintf(stderr, "usage: ols [-s] [-g] filename\n");
    fprintf(stderr, "       -s: select stupid greedy algorithm\n");
    fprintf(stderr, "       -g: also output the rows to be bribed\n");
    return 1;
  }

  // read from file
  struct timeval startReadTime;
  gettimeofday(&startReadTime, NULL);
  
  FILE* f = fopen(argv[1], "r");
  assert(f);
  
  int k;
  fscanf(f, "# k=%d,", &k);
  
  cMatrix matrix(read_matrix_from_file(f));
  
  fclose(f);
  
  matrix.computeGaps();
  //fprintf(stderr, "input data:\n");
  //matrix.dump();
  if(gossip){
    printf("Maximum gap and the gap values of each column:\n");
  }
  matrix.gaps();
  
  struct timeval startTime;
  gettimeofday(&startTime, NULL);
  const list<unsigned> bribes = useStupid ? stupid(matrix) : greedy(matrix);
  struct timeval endTime;
  gettimeofday(&endTime, NULL);
  struct timeval solve_duration, total_duration;
  timersub(&endTime, &startTime, &solve_duration);
  timersub(&endTime, &startReadTime, &total_duration);
  //  fprintf(stderr, "took %lu microseconds\n",
  //	  duration.tv_sec * 1000000UL + duration.tv_usec); 
  //  fprintf(stderr, "greedy algorithm needs %lu bribes\n", bribes.size());
  
  if (gossip){
    fprintf(stdout, "%lu bribes, the rows to be bribed:\n", bribes.size());
    list<unsigned>::const_iterator rowIter;
    for(rowIter=bribes.begin(); rowIter != bribes.end(); ++rowIter){
      fprintf(stdout, "%u ", *rowIter);
    }
    printf("\n");
    return 0;
  }
  
  if (k != bribes.size()){
    //      fprintf(stderr, "Greedy alg. not optimal for matrix\n");
    //cMatrix copy(matrix.getNRows(), matrix.getNCols());
    //create_matrix(copy, mtype, s, matrix.getNCols());
    // cMatrix copy(matrix);
    // copy.computeGaps();
    //copy.dump();
    fprintf(stdout, "GD needs %lu bribes while optimal needs %d ones\n",
	    bribes.size(), k);
  }
  
  // output for script: optimal greedy duration (in microseconds)
  // bribe_size, k, solve_duration, total_duration
  fprintf(stdout, "%lu %d %f %f\n",
	  bribes.size(), k, 
	  solve_duration.tv_sec+(solve_duration.tv_usec)/1000000.0,
	  total_duration.tv_sec+(total_duration.tv_usec)/1000000.0);
  
  return 0;
}
